/******************************************************************************
  @file    unisoc.c
  @brief   unisoc log tool.

  DESCRIPTION
  QLog Tool for USB and PCIE of Quectel wireless cellular modules.

  INITIALIZATION AND SEQUENCING REQUIREMENTS
  None.

  ---------------------------------------------------------------------------
  Copyright (c) 2016 - 2020 Quectel Wireless Solution, Co., Ltd.  All Rights Reserved.
  Quectel Wireless Solution Proprietary and Confidential.
  ---------------------------------------------------------------------------
******************************************************************************/

#include "qlog.h"

//#define UNISOC_DUMP_TEST

typedef uint8_t   BYTE;
typedef uint32_t  DWORD;

/* DIAG */
#define DIAG_SWVER_F             0                   // Information regarding MS software
#define DIAG_SYSTEM_F            5
#define MSG_BOARD_ASSERT         255

#define ITC_REQ_TYPE	         209
#define ITC_REQ_SUBTYPE          100
#define ITC_REP_SUBTYPE          101

#define UET_SUBTYPE_CORE0        102                 // UE Time Diag SubType field, core0
#define TRACET_SUBTYPE           103                 // Trace Time Diag subtype filed
#define UET_SUBTYPE_CORE1        104                 // UE Time Diag SubType field, core1

#define UET_SEQ_NUM              0x0000FFFF          // UE Time Diag Sequnce number fi
#define UET_DIAG_LEN             (DIAG_HDR_LEN + 12) // UE Time Diag Length field (20)

/* ASSERT */
#define NORMAL_INFO              0x00
#define DUMP_MEM_DATA            0x01
#define DUMP_ALL_ASSERT_INFO_END 0x04
#define DUMP_AP_MEMORY           0x08

#define HDLC_HEADER 0x7e

//Export type definitions
#define DIAG_HDR_LEN     8
typedef struct 
{
    unsigned int sn;			// Sequence number
    unsigned short len;			// Package length
    unsigned char type;
    unsigned char subtype;
} DIAG_HEADER;

typedef struct 
{
    DIAG_HEADER header;
    uint8_t data[0];					// Package data
}DIAG_PACKAGE;

// SMP package
#define SMP_START_FLAG  0x7E7E7E7E
#define SMP_START_LEN   4
#define SMP_HDR_LEN     8
#define SMP_RESERVED    0x5A5A
typedef struct
{
    unsigned short len;
    unsigned char  lcn;
    unsigned char  packet_type;
    unsigned short reserved;    //0x5A5A
    unsigned short check_sum;
} SMP_HEADER;

typedef struct 
{
    SMP_HEADER header;
    uint8_t data[0];					// Package data
} SMP_PACKAGE;

static int m_bUE_Obtained  = 0;
static int m_bVer_Obtained = 0;
static int m_bSendTCmd = 0;

#define unisoc_log_max (24*1024)
static uint8_t *unisoc_log_buf = NULL;
static size_t unisoc_log_size = 0;

static size_t diag_pkt_size = (64*1024);
static uint8_t *diag_pkt_buf = NULL;
static uint8_t *diag_pkt_start = NULL;
static uint8_t *diag_pkt_end = NULL;
static uint8_t *diag_pkt_cur = NULL;
static int diag_last_7d = 0;

static int unisoc_normal_fd = -1;
static int g_logfd = 0;

static uint8_t _g_log_type = 0; // 0b01 log, 0b10 dump, 0b11 log&dump
#define LOG_TYPE_LOG  0b01
#define LOG_TYPE_DUMP 0b10

static void hexdump(const uint8_t *d, size_t size) {
    unsigned i;
    unsigned n=64;

    printf("%s size=%zd\n", __func__, size);
    for (i = 0; i < size; i++) {
        if (i%n==0)
            printf("%04d:", i);
        printf(" %02x", d[i]);
        if (i%n==(n-1))
            printf("\n");
    }

    printf("\n");
}

static unsigned short frm_chk( unsigned short *src, int len,int nEndian )
{
    unsigned int sum = 0;
    unsigned short SourceValue, DestValue;
    unsigned short lowSourceValue, hiSourceValue;

    /* Get sum value of the source.*/
    while (len > 1)
    {     
        SourceValue = *src++;
        if( nEndian == 1 )
		{
			// Big endian
			DestValue   = 0;      
			lowSourceValue = (unsigned short)(( SourceValue & 0xFF00 ) >> 8);
			hiSourceValue = (unsigned short)(( SourceValue & 0x00FF ) << 8);
			DestValue = (unsigned short)(lowSourceValue | hiSourceValue);
		}
		else
		{
			// Little endian
			DestValue = qlog_le16(SourceValue);
		}
        sum += DestValue;	
        len -= 2;
    }

    if (len == 1)
    {
        sum += *( (unsigned char *) src );
    }

    sum = (sum >> 16) + (sum & 0x0FFFF);
    sum += (sum >> 16);

    return (unsigned short)(~sum);   
}

static int CheckHeader(SMP_HEADER *pHdr)
{
	if( frm_chk((unsigned short*)pHdr, SMP_HDR_LEN, 0) == 0)
		return 1;

	return 0;
}

/* get sys timestamp */
static long GetTimeStamp()
{
    struct timeval tm;
    gettimeofday(&tm, NULL);

    return (tm.tv_sec * 1000 + tm.tv_usec / 1000);
}

static void WriteDataToFile(int logfd, const void *lpBuffer, DWORD dwDataSize)
{
    //printf("%s size=%d\n", __func__, dwDataSize);
#ifndef UNISOC_DUMP_TEST
    uint32_t le32 = qlog_le32(dwDataSize);
    write(logfd, &le32, sizeof(dwDataSize));
    write(logfd, lpBuffer, dwDataSize);
#endif
}

static int unisoc_diag_fd = -1;

static int unisoc_send_cmd(uint8_t cmd) {
    uint8_t lpBuf[2] = {cmd, '\n'};

    qlog_dbg("%s cmd='%c'\n", __func__, cmd);
    return write(unisoc_diag_fd, lpBuf, 2);
}

static int unisoc_init_filter(int fd, const char *cfg) {
    uint8_t lpBuf[32] = {0};
    DIAG_HEADER request;

    unisoc_diag_fd = fd;
    unisoc_send_cmd('0');

    while (m_bUE_Obtained == 0 || m_bVer_Obtained == 0)
    {
        if (m_bUE_Obtained == 0) {
            // query modem UE time
            request.len         = qlog_le16(DIAG_HDR_LEN);
            request.type        = DIAG_SYSTEM_F;
            request.subtype     = 0x11;
            request.sn          = qlog_le32(0);

            lpBuf[0] = 0x7E;
            memcpy(lpBuf+1, &request, DIAG_HDR_LEN);
            lpBuf[DIAG_HDR_LEN+1] = 0x7E;

            write(fd, lpBuf, DIAG_HDR_LEN+2);
        }

        if (m_bVer_Obtained == 0) {
            // query modem version
            request.len         = qlog_le16(DIAG_HDR_LEN);
            request.type        = DIAG_SWVER_F;
            request.subtype     = 0x0;
            request.sn          = qlog_le32(0);

            lpBuf[0] = 0x7E;
            memcpy(lpBuf+1, &request, DIAG_HDR_LEN);
            lpBuf[DIAG_HDR_LEN+1] = 0x7E;    

            write(fd, lpBuf, DIAG_HDR_LEN+2);
        }

        sleep(5);

#ifdef UNISOC_DUMP_TEST
        if (1) {
            uint8_t asser_modem[] = {0x7e, 0x00, 0x00, 0x00,  0x00, 0x08, 0x00, 0x05, 0x04, 0x7e};
            //uint8_t asser_ap[] = {0x7e, 0x00, 0x00, 0x00,  0x00, 0x08, 0x00, 0x05, 0x08, 0x7e};

            write(fd, asser_modem, sizeof(asser_modem));
            break;
        }
#endif
    }

    return 0;
}

static int unisoc_logfile_init(int logfd, unsigned logfile_seq) {
    // write first reserved package
    BYTE lpBuf[32] = {0};
    DIAG_HEADER ph = {0};

    ph.sn      = qlog_le32(0);
    ph.type    = ITC_REQ_TYPE;
    ph.subtype = ITC_REP_SUBTYPE;
    ph.len     = qlog_le16(DIAG_HDR_LEN + 8);

    memcpy(lpBuf, &ph, DIAG_HDR_LEN);
    ((DWORD*)(lpBuf + DIAG_HDR_LEN ))[0] = qlog_le32(7);
    lpBuf[ DIAG_HDR_LEN + 4 ] = 0; //0:Little Endian,1: Big Endian
    lpBuf[ DIAG_HDR_LEN + 5 ] = 1;
    lpBuf[ DIAG_HDR_LEN + 6 ] = 1;

    WriteDataToFile(logfd, lpBuf, DIAG_HDR_LEN + 8);

    // write default sys time package
    memset(&ph, 0, sizeof( DIAG_HEADER));
    ph.sn      = qlog_le32(UET_SEQ_NUM);
    ph.type    = ITC_REQ_TYPE;
    ph.subtype = UET_SUBTYPE_CORE0;
    ph.len     = qlog_le16(UET_DIAG_LEN);

    memcpy(lpBuf, &ph, DIAG_HDR_LEN);
    ((uint64_t *)(lpBuf + DIAG_HDR_LEN ))[0] = qlog_le64(GetTimeStamp());
    ((uint32_t *)(lpBuf + DIAG_HDR_LEN ))[2]  = qlog_le32(0);

    WriteDataToFile(logfd, lpBuf, UET_DIAG_LEN);

    return 0;
}

static void unisoc_proc_log_buf(int logfd, const void *inBuf, size_t size) {
    const uint8_t *lpBuf = inBuf;
    size_t cur = 0;
    SMP_PACKAGE* lpPkg;
    unsigned char  agStartFlag[SMP_START_LEN] = {0x7E,0x7E,0x7E,0x7E};

    if (unisoc_log_buf == NULL) {
        unisoc_log_buf = (uint8_t *)malloc(unisoc_log_max);
    }
    if (unisoc_log_buf == NULL) {
        return;
    }
    //hexdump(buf, size);

    //printf("%s cur=%zd, add=%zd\n", __func__, unisoc_log_size, size);
    if (unisoc_log_size) {
        if ((unisoc_log_size + size) > unisoc_log_max) {
            printf("%s cur=%zd, add=%zd\n", __func__, unisoc_log_size, size);
            return;
        }
        memcpy(unisoc_log_buf + unisoc_log_size, inBuf, size);
        size += unisoc_log_size;
        inBuf = unisoc_log_buf;
    }

    while (cur < size) {
        uint16_t pkt_len;
        lpBuf = inBuf + cur;

        if ((cur + SMP_START_LEN + sizeof(SMP_HEADER)) > size) {
            break;
        }

        if (memcmp(lpBuf, agStartFlag, SMP_START_LEN)) {
            if (cur == 0) {
                //printf("%s agStartFlag %02x%02x%02x%02x\n", __func__, lpBuf[0], lpBuf[1], lpBuf[2], lpBuf[3]);
                //hexdump(lpBuf, MIN(100, size-cur));
            }
            cur++;
            if (cur == size) {
                cur--;
                break;
            }
            continue;
        }

        lpPkg = ( SMP_PACKAGE *)(lpBuf + SMP_START_LEN);

        if (CheckHeader(&lpPkg->header) == 0) {
            printf("%s CheckHeader fail\n", __func__);
            hexdump(lpBuf, MIN(100, size-cur));
            cur++;
            continue;
        }

        if (lpPkg->header.lcn != 0) {
            printf("%s header.lcn=0x%x\n", __func__, lpPkg->header.lcn);
            hexdump(lpBuf, MIN(100, size-cur));
            cur++;
            continue;
        }

        if (lpPkg->header.packet_type != 0 && lpPkg->header.packet_type != 0xf8) {
            printf("%s header.packet_type=0x%x\n", __func__, lpPkg->header.packet_type);
            hexdump(lpBuf, MIN(100, size-cur));
            cur++;
            continue;
        }

        if (qlog_le16(lpPkg->header.reserved) != 0x5A5A) {
            printf("%s header.reserved=0x%x\n", __func__, qlog_le16(lpPkg->header.reserved));
            hexdump(lpBuf, MIN(100, size-cur));
            cur++;
            continue;
        }

        pkt_len = qlog_le16(lpPkg->header.len);
        if ((cur + SMP_START_LEN + pkt_len) > size) {
            //printf("%s wait for more data, cur=%zd, size=%zd, len=%d\n", __func__, cur, size, lpPkg->header.len);
            break;
        }

        cur += (SMP_START_LEN + pkt_len);
        WriteDataToFile(logfd, lpBuf + SMP_START_LEN + sizeof(SMP_HEADER), pkt_len - sizeof(SMP_HEADER));
    }

    unisoc_log_size = size - cur;
    if (unisoc_log_size)
        memmove(unisoc_log_buf, lpBuf, unisoc_log_size);
    //printf("%s left=%zd\n", __func__, unisoc_log_size);
}

static void HandleAssertProc(DIAG_PACKAGE* lpPkg) {

    //printf("%s subtype=%d, len=%d\n", __func__, lpPkg->header.subtype, lpPkg->header.len);

    if (NORMAL_INFO == lpPkg->header.subtype)
    {
        if (unisoc_normal_fd == -1) {
            unisoc_normal_fd = qlog_create_file_in_logdir("normar.txt");
        }
        if (unisoc_normal_fd != -1) {
            write(unisoc_normal_fd, lpPkg->data, qlog_le32(lpPkg->header.len));
        }
        
        //qlog_dbg("%s NORMAL_INFO\n", __func__);
        //printf("%.*s\n", qlog_le32(lpPkg->header.len), lpPkg->data);
        if (!m_bSendTCmd) {
            m_bSendTCmd = 1;
            unisoc_send_cmd('t');
        }
    }
    else if (DUMP_MEM_DATA == lpPkg->header.subtype) // recv mem data
    {
        static int show_once = 0;
        /**
         * OOPS: modem come into DUMP when catching DUMP
         * remember, the logfd is still the one for saving log
         * so close log file and create an new dump file
         */
        if (_g_log_type & LOG_TYPE_LOG) {
            int fd = qlog_create_file_in_logdir ("first_dump_mem.bin");
            dup2(fd, g_logfd);
            unisoc_logfile_init (g_logfd, 0);
        }

        if (!show_once) {
            show_once = 1;
            qlog_dbg("you see this message, because the modem come in dump!");
        }
    }
    else if (DUMP_ALL_ASSERT_INFO_END == lpPkg->header.subtype) // recv assert end
    {
        qlog_dbg("%s DUMP_ALL_ASSERT_INFO_END\n", __func__);
        unisoc_send_cmd('z'); //Reset MCU
    }
    else
   {
        qlog_dbg("%s subtype=%d\n", __func__, lpPkg->header.subtype);
   }
}

static uint32_t last_diag_cmd_type = 0;
static int HandleDiagPkt(const uint8_t *lpBuf, size_t size) {
    uint16_t pkt_len;
    DIAG_PACKAGE* lpPkg;

    lpPkg = (DIAG_PACKAGE *)(lpBuf + 1);

    pkt_len = qlog_le16(lpPkg->header.len);

    if (pkt_len > 0x4008) {
        qlog_dbg("larger pkt_len= %d\n", pkt_len);
    }

    if (pkt_len != ( size - 2)) {
        if (pkt_len)
            qlog_dbg("fix pkt_len= %d -> %zd\n", pkt_len, size - 2);
        pkt_len = size - 2;
        lpPkg->header.len = qlog_le16(pkt_len);
    }

    if (DIAG_SYSTEM_F == lpPkg->header.type && 0x11 == lpPkg->header.subtype)
    {
        m_bUE_Obtained = 1;
    }
    else if (DIAG_SWVER_F == lpPkg->header.type && 0x0 == lpPkg->header.subtype)
    {
        m_bVer_Obtained = 1;
    }
    else if (DIAG_SYSTEM_F == lpPkg->header.type && 0x1 == lpPkg->header.subtype)
    {
    }
    else if (MSG_BOARD_ASSERT == lpPkg->header.type)
    {
        HandleAssertProc(lpPkg);
    }
    else 
    {
        qlog_dbg("%s unknow type subtype\n", __func__);
        hexdump(lpBuf, size);
    }
    last_diag_cmd_type = (lpPkg->header.type << 8) + lpPkg->header.subtype;

    return 1;
}

static void unisoc_proc_diag_buf(int logfd, const void *inBuf, size_t size) {
    const uint8_t *pSrc = inBuf;
    size_t i;

#ifdef UNISOC_DUMP_TEST
    static int unisoc_raw_diag_fd = -1;
    if (unisoc_raw_diag_fd == -1) {
        unisoc_raw_diag_fd = qlog_create_file_in_logdir("../raw_diag.bin");
    }
    if (unisoc_raw_diag_fd  != -1) {
        write(unisoc_raw_diag_fd , inBuf, size);
    }
#endif

    if (diag_pkt_buf == NULL) {
        diag_pkt_buf = (uint8_t *)malloc(diag_pkt_size);
        diag_pkt_cur = diag_pkt_buf;
    }

    if (diag_pkt_buf == NULL) {
        return;
    }
   
    if ((diag_pkt_cur - diag_pkt_buf + size) > diag_pkt_size) {
        printf("%s cur=%zd, add=%zd\n", __func__, diag_pkt_cur - diag_pkt_buf , size);
        diag_pkt_cur = diag_pkt_buf;
        diag_pkt_start = diag_pkt_end = NULL;
        return;
    }

    if (diag_last_7d) {
        *diag_pkt_cur++ = (*pSrc++ ^ 0x20);
        size--;
        diag_last_7d = 0;
    }

    for (i = 0; i < size; i++) {
        if (*pSrc == 0x7d) {
            pSrc++;
            i++;
            if (i == size) {
                //qlog_dbg("last_7d\n");
                diag_last_7d = 1;
                break;
            }
            *diag_pkt_cur++ = (*pSrc++ ^ 0x20);
        }
        else if (*pSrc == 0x7E) {
            if (diag_pkt_start == NULL) {
                diag_pkt_start = diag_pkt_cur;
            }
            else {
                diag_pkt_end = diag_pkt_cur;
            }
            *diag_pkt_cur++ = *pSrc++;

            if (diag_pkt_start && diag_pkt_end) {
                if (diag_pkt_start < diag_pkt_end) {
                    size_t pkt_len = diag_pkt_end - diag_pkt_start + 1;
                    if (pkt_len >= (sizeof(DIAG_PACKAGE) + 2)) {
                        if (HandleDiagPkt(diag_pkt_start, pkt_len)) {
                            WriteDataToFile(logfd, diag_pkt_start+1, pkt_len-2);
                        }
                    }
                    else {
                        qlog_dbg("diag_pkt start=%zd, end=%zd, len=%zd, last read size=%zd\n",
                            diag_pkt_start - diag_pkt_buf, diag_pkt_end - diag_pkt_buf, pkt_len, size);
                        exit(0);
                    }
                }
                else {
                    qlog_dbg("diag_pkt start=%zd, end=%zd, last read size=%zd\n",
                            diag_pkt_start - diag_pkt_buf, diag_pkt_end - diag_pkt_buf, size);
                    exit(0);
               }

                diag_pkt_cur = diag_pkt_start;
                diag_pkt_start = diag_pkt_end = NULL;
            }
        }
        else if (diag_pkt_start == NULL) {
            if (last_diag_cmd_type != (MSG_BOARD_ASSERT << 8) + DUMP_MEM_DATA) {
                
            }
            else {
                qlog_dbg("%s should get 0x7e here! last read size=%zd, i=%zd\n", __func__, size, i);
                hexdump(inBuf, size);
                exit(0);
            }
            pSrc++;
        }
        else {
            *diag_pkt_cur++ = *pSrc++;
        }
    }
}

extern int g_unisoc_log_type;
static size_t unisoc_logfile_save(int logfd, const void *buf, size_t size) {
    g_logfd = logfd;
    if (g_unisoc_log_type == 1) {
        _g_log_type |= LOG_TYPE_LOG;
        unisoc_proc_log_buf(logfd, buf, size);
    }
    else if (g_unisoc_log_type == 0)
        unisoc_proc_diag_buf(logfd, buf, size);
    return size;
}

qlog_ops_t unisoc_qlog_ops = {
    .init_filter = unisoc_init_filter,
    //.clean_filter = mdm_clean_filter,
    .logfile_init = unisoc_logfile_init,
    .logfile_save = unisoc_logfile_save,
};
