/******************************************************************************
  @file    mdm.c
  @brief   mdm log tool.

  DESCRIPTION
  QLog Tool for USB and PCIE of Quectel wireless cellular modules.

  INITIALIZATION AND SEQUENCING REQUIREMENTS
  None.

  ---------------------------------------------------------------------------
  Copyright (c) 2016 - 2020 Quectel Wireless Solution, Co., Ltd.  All Rights Reserved.
  Quectel Wireless Solution Proprietary and Confidential.
  ---------------------------------------------------------------------------
******************************************************************************/

#include "qlog.h"

static const uint8_t *g_mdm_req;
#include "qshrink4.c"

/*
80-na157-61_yb_diagnostic_system_user_guide.pdf
For non-HDLC encoded config
When .cfg2 is used, which is non-HDLC encoded, the format of the file is as follows:
Field                              Length (in bits) Description
------- ---------------- -----------
Start                              8  This is the start of packet, 0x7E
Version                          8  Version
Payload Length              16  Payload length
Payload Variable This is the actual data
Packet End                    8 Ending character "0x7E"

1d 1c 3b 7e // 0x1D == Time Stamp Request
00 78 f0 7e // 0x00 == Version Request
7c 93 49 7e // 0x7C == Extended Build ID Request
1c 95 2a 7e // 0x1C == DIAG Version Request
0c 14 3a 7e // 0x0c == Status Request
63 e5 a1 7e // 0x6C == Phone State Request

4.7.5 Reading the CFG file
Open the CFG file in a hex editor.
For HDLC-encoded config
Field                             Length (in bits) Description
------- ---------------- -----------
Information Variable ICD Packet or Message
Frame Check                16 CRC-CCITT standard 16-bit CRC
Ending Flag                   8 Ending character "0x7E"

4b 0f 1a 00 00 bb 60 7e // 0x4B 0x0F 0x001A == Call Manager Subsystem Sys Select (80-V1294-7)
4B 09 00 00 62 B6 7E // 0x4B 0x09 0x0000 == UMTS Subsystem Version Request (80-V2708-1
4b 08 00 00 be ec 7e // 0x4B 0x08 0x0000 == GSM Subsystem Version Request (80-V5295-1)
4b 08 01 00 66 f5 7e // 0x4B 0x0F 0x0001 == GSM Subsystem Status Request (80-V5295-1)
4b 04 00 00 1d 49 7e // 0x4B 0x04 0x0000 == WCDMA Subsystem Version Request (80-V2708-1)
4b 04 0f 00 d5 ca 7e // 0x4B 0x04 0x000F == WCDMA Subsystem Additional Status Request (80-V2708-1)

80-v1294-100_c_mask-related_commands_for_diagnostic_monitoring.pdf
2.6 Extended Message Configuration (125 / 0x7D)
2.4 Logging Configuration (115 / 0x73)
2.2 Event Report Control (96 / 0x60)
2.8 Event Set Mask (130 / 0x82)

C:\ProgramData\QUALCOMM\QXDM\Config\Qualcomm DMC Library\Primary\Default.cfg
*/
static unsigned char qlog_mdm_default_cfg[] = {
0x1d,0x1c,0x3b,0x7e,0x00,0x78,0xf0,0x7e,0x4b,0x32,0x06,0x00,0xba,0x4d,0x7e,0x7c,0x93,0x49,0x7e,0x1c,0x95,0x2a,0x7e,0x0c,0x14,0x3a,0x7e,0x63,0xe5,0xa1,0x7e,0x4b,
0x0f,0x00,0x00,0xbb,0x60,0x7e,0x4b,0x09,0x00,0x00,0x62,0xb6,0x7e,0x4b,0x08,0x00,0x00,0xbe,0xec,0x7e,0x4b,0x08,0x01,0x00,0x66,0xf5,0x7e,0x4b,0x04,0x00,0x00,0x1d,
0x49,0x7e,0x4b,0x04,0x0f,0x00,0xd5,0xca,0x7e,0x4b,0x0f,0x18,0x00,0x01,0x9e,0xa9,0x7e,0x4b,0x0f,0x18,0x00,0x02,0x05,0x9b,0x7e,0x4b,0x0f,0x2c,0x00,0x28,0xea,0x7e,
0x4b,0x12,0x39,0x00,0xeb,0x7b,0x7e,0x4b,0x12,0x3c,0x00,0x53,0x05,0x7e,0x4b,0x12,0x37,0x00,0xfb,0xe1,0x7e,0x4b,0x12,0x3b,0x00,0x5b,0x48,0x7e,0x4b,0x12,0x35,0x00,
0x4b,0xd2,0x7e,0x4b,0x12,0x3a,0x00,0x83,0x51,0x7e,0x4b,0x12,0x00,0x08,0x19,0x96,0x7e,0x7d,0x5d,0x05,0x00,0x00,0x00,0x00,0x00,0x00,0x74,0x41,0x7e,0x7d,0x5d,0x04,
0x00,0x00,0x02,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1c,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x4c,0x06,0x7e,0x7d,0x5d,0x04,0x05,0x00,0x05,0x00,0x00,0x00,0x1f,0x00,
0x00,0x00,0xce,0xa7,0x7e,0x7d,0x5d,0x04,0x07,0x00,0x08,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0xd0,0x71,0x7e,0x7d,0x5d,0x04,0x0b,0x00,0x0c,0x00,
0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x7c,0x68,0x7e,0x7d,0x5d,0x04,0x0e,0x00,0x12,0x00,0x00,0x00,0xff,0x01,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,
0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0xb5,0x3a,0x7e,0x7d,0x5d,0x04,0x14,0x00,0x15,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x2e,0xbb,
0x7e,0x7d,0x5d,0x04,0x19,0x00,0x19,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x88,0xd0,0x7e,0x7d,0x5d,0x04,0x20,0x00,0x20,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0xea,0xaa,
0x7e,0x7d,0x5d,0x04,0x27,0x00,0x28,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0xfe,0x01,0x00,0x00,0x89,0x11,0x7e,0x7d,0x5d,0x04,0x2a,0x00,0x2b,0x00,0x00,0x00,0x1e,0x00,
0x00,0x00,0x1e,0x00,0x00,0x00,0x01,0x1a,0x7e,0x7d,0x5d,0x04,0x33,0x00,0x33,0x00,0x00,0x00,0x1f,0x00,0x00,0x00,0xc2,0xc1,0x7e,0x7d,0x5d,0x04,0x36,0x00,0x36,0x00,
0x00,0x00,0x1e,0x00,0x00,0x00,0xa3,0xd6,0x7e,0x7d,0x5d,0x04,0x39,0x00,0x3a,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x65,0xdf,0x7e,0x7d,0x5d,0x04,
0x3f,0x00,0x41,0x00,0x00,0x00,0x1f,0x00,0x00,0x00,0xfe,0xff,0x1f,0x00,0x1f,0x00,0x00,0x00,0xc9,0x67,0x7e,0x7d,0x5d,0x04,0x44,0x00,0x45,0x00,0x00,0x00,0x1e,0x00,
0x00,0x00,0x1e,0x00,0x00,0x00,0x2e,0x6c,0x7e,0x7d,0x5d,0x04,0x48,0x00,0x4a,0x00,0x00,0x00,0x1f,0x00,0x00,0x00,0x1f,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x4f,0xf2,
0x7e,0x7d,0x5d,0x04,0x4c,0x00,0x4c,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0xf2,0x66,0x7e,0x7d,0x5d,0x04,0x4e,0x00,0x4e,0x00,0x00,0x00,0x07,0x00,0x00,0x00,0x11,0x0f,
0x7e,0x7d,0x5d,0x04,0x58,0x00,0x58,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x9a,0x49,0x7e,0x7d,0x5d,0x04,0x5a,0x00,0x5b,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,
0x00,0x00,0x0e,0x43,0x7e,0x7d,0x5d,0x04,0x63,0x00,0x63,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0xd9,0x60,0x7e,0x7d,0x5d,0x04,0x70,0x00,0x70,0x00,0x00,0x00,0x1e,0x00,
0x00,0x00,0x4a,0x17,0x7e,0x7d,0x5d,0x04,0x75,0x00,0x75,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0xc8,0x6d,0x7e,0x7d,0x5d,0x04,0xea,0x03,0xea,0x03,0x00,0x00,0x1e,0x00,
0x00,0x00,0xa9,0x9e,0x7e,0x7d,0x5d,0x04,0xee,0x03,0xef,0x03,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x9b,0x20,0x7e,0x7d,0x5d,0x04,0xd0,0x07,0xd7,0x07,
0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,
0x00,0x00,0xdc,0x10,0x7e,0x7d,0x5d,0x04,0xb8,0x0b,0xc5,0x0b,0x00,0x00,0x1f,0x00,0x00,0x00,0xfe,0xff,0x7f,0x00,0x7f,0x00,0x00,0x00,0x1f,0x00,0x00,0x00,0x1f,0x00,
0x00,0x00,0xff,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,
0x00,0x00,0x1c,0x00,0x00,0x00,0xa9,0x36,0x7e,0x7d,0x5d,0x04,0xa0,0x0f,0xaa,0x0f,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,
0x00,0x00,0xfe,0x01,0x00,0x00,0x1e,0x00,0x00,0x00,0xfe,0xff,0x01,0x00,0xfe,0xff,0x07,0x00,0xfe,0xff,0x01,0x00,0xfe,0x07,0x00,0x00,0x1e,0x00,0x00,0x00,0xc3,0xb9,
0x7e,0x7d,0x5d,0x04,0x05,0x12,0x05,0x12,0x00,0x00,0x1f,0x00,0x00,0x00,0xd2,0x41,0x7e,0x7d,0x5d,0x04,0x07,0x12,0x07,0x12,0x00,0x00,0x1f,0x00,0x00,0x00,0xf3,0x12,
0x7e,0x7d,0x5d,0x04,0x88,0x13,0xa8,0x13,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,
0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,
0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1f,0x00,0x00,0x00,0x1f,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,
0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,
0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0xf4,0x4c,0x7e,0x7d,0x5d,0x04,0x72,0x17,0x72,0x17,0x00,0x00,0x1e,0x00,0x00,0x00,0xcc,0x20,
0x7e,0x7d,0x5d,0x04,0x74,0x17,0x74,0x17,0x00,0x00,0x3f,0x00,0x00,0x00,0x47,0x46,0x7e,0x7d,0x5d,0x04,0x93,0x17,0x93,0x17,0x00,0x00,0x1e,0x00,0x00,0x00,0x8f,0xca,
0x7e,0x7d,0x5d,0x04,0x97,0x17,0x97,0x17,0x00,0x00,0x1e,0x00,0x00,0x00,0xcd,0x6c,0x7e,0x7d,0x5d,0x04,0xa4,0x17,0xb7,0x17,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,
0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,
0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,
0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x8d,0xd1,0x7e,0x7d,0x5d,0x04,0xc0,0x17,0xc0,0x17,0x00,0x00,0x1e,0x00,0x00,0x00,0x96,0x89,0x7e,0x7d,0x5d,0x04,
0x34,0x21,0x34,0x21,0x00,0x00,0x1e,0x00,0x00,0x00,0x10,0xc3,0x7e,0x7d,0x5d,0x04,0x1c,0x25,0x25,0x25,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0xff,0x1f,0x00,0x7d,0x5e,
0x00,0x00,0x00,0x3e,0x00,0x00,0x00,0x7d,0x5e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0xfe,0x03,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x3e,0xd0,0x00,0x00,
0x90,0xed,0x7e,0x7d,0x5d,0x04,0x0b,0x28,0x0f,0x28,0x00,0x00,0x1c,0x00,0x00,0x00,0x1c,0x00,0x00,0x00,0x1c,0x00,0x00,0x00,0x1c,0x00,0x00,0x00,0x1c,0x00,0x00,0x00,
0x71,0x86,0x7e,0x7d,0x5d,0x04,0x6e,0x28,0x89,0x28,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,
0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,
0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,
0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x1e,0x00,0x00,0x00,0x52,0x90,0x7e,0x73,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xda,0x81,0x7e,0x73,0x00,0x00,0x00,0x03,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0xfe,0x0f,0x00,0x00,0xf0,0x07,0xc8,0x00,0x00,0x40,
0xc4,0x00,0x00,0x00,0x00,0xc0,0x49,0xf3,0xc7,0x5b,0x7c,0xf3,0x0b,0x01,0x00,0x00,0x00,0x20,0xec,0x00,0xcc,0x83,0x01,0x00,0x00,0x00,0x00,0x00,0x38,0x00,0x38,0x00,
0x38,0x00,0x38,0x00,0x00,0x01,0x01,0x00,0x40,0x08,0xf0,0x07,0x0c,0xf8,0x47,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x40,0x04,0x00,0xff,0xf7,0x7f,0xf0,0xfc,0xff,0xff,0xad,0xe0,0x7f,0x02,0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x78,
0xe0,0xff,0xff,0xff,0x48,0x1c,0x1e,0x00,0x03,0x10,0x18,0xff,0xff,0xff,0xff,0xbf,0x00,0x00,0x00,0x00,0x00,0x90,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x80,0x1b,0x80,0xff,0x5f,0x06,0x00,0x00,0x41,0x00,0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x04,0xc0,0x07,0x01,0x00,0x30,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x10,0x00,0x80,0x00,0x00,0x7f,0xce,0x00,0x00,0x01,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x18,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0c,0x00,0x00,
0x08,0x00,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0xf8,0x07,0x00,0x00,0x00,
0x07,0x00,0x00,0xc0,0x7f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x06,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0a,0x30,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x40,0x00,0x00,0x00,0x00,0x06,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x20,0xb7,0xb2,0x7e,0x73,0x00,0x00,
0x00,0x03,0x00,0x00,0x00,0x04,0x00,0x00,0x00,0x09,0x08,0x00,0x00,0x31,0x00,0x09,0x80,0xf8,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7f,0xef,0x20,0xf0,0x90,0x3c,0x1d,0x60,0x04,0x00,0x00,0x4f,0x03,0xfe,0x07,0x43,0x0b,0x02,
0x01,0x00,0x00,0x07,0xf4,0x45,0x00,0x38,0x00,0x00,0x00,0x00,0x00,0x86,0x8a,0x45,0xf8,0x25,0x10,0x00,0xcc,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x82,0xc3,0x1e,0x00,0x7d,0x5e,0x00,0x4e,0x00,0xff,0x03,0x00,0x3c,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x09,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xff,0x01,0x32,0x3e,0x7e,0x73,0x00,0x00,0x00,0x03,0x00,0x00,0x00,0x05,0x00,0x00,0x00,0x5d,
0x0c,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x70,0x90,0x6f,0x3b,0xfc,0x01,0x13,0x00,0x00,0x00,0x00,0x00,0x00,0x07,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xd0,0xff,0xd7,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0xf0,0xff,0xe7,0xff,0xbf,0xf3,0x43,0x3f,0x02,0x00,0xe0,0xe3,0x01,0xff,0x0d,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x70,0x90,0x6f,0x1f,0xfc,0x71,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x07,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xd0,0xff,0x57,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0xe0,0xe7,0xff,0xf7,0xc3,0x3f,0x01,0x00,0xe0,0xe3,0x01,0x10,0x5d,0xe6,0x7e,0x73,0x00,0x00,0x00,0x03,0x00,0x00,0x00,0x07,0x00,0x00,0x00,0x57,0x0b,
0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xff,0xff,0x01,0x00,0x7f,0x02,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0xff,0x37,0x06,0xea,0x07,0x00,0x07,0x00,0x00,0x00,0x00,0xfc,0xf0,0x20,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xff,0x01,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xff,0x77,0x00,0x00,0x7f,0x9f,0xaa,0x7e,0x73,0x00,0x00,0x00,0x03,0x00,0x00,0x00,0x0b,0x00,0x00,0x00,0xd0,0x01,0x00,0x00,
0x06,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x5f,0x00,0x00,0x00,0xce,0x04,0xce,0x00,0x3f,0x00,0x3f,0x00,0xdf,0x24,0x00,0x00,0x7f,0xfc,0x3c,0x00,
0x00,0x00,0x3e,0x28,0x4e,0x50,0x05,0x12,0x51,0xe0,0x00,0x00,0xff,0xff,0xff,0xe2,0xcf,0xe1,0x7d,0x5d,0x51,0x7f,0x00,0x01,0x74,0x42,0xe0,0xac,0x34,0x7e,0x73,0x00,
0x00,0x00,0x03,0x00,0x00,0x00,0x0d,0x00,0x00,0x00,0x39,0x01,0x00,0x00,0x79,0xa6,0xff,0xff,0xff,0xa1,0x1f,0x00,0xdf,0x01,0x00,0x00,0x03,0x40,0x00,0x00,0x00,0x9f,
0x00,0x00,0x00,0x00,0xc0,0x06,0x00,0x00,0x00,0x00,0x5f,0x00,0x00,0x00,0xff,0x37,0xff,0xbf,0x1b,0x05,0x00,0x01,0x2b,0x17,0x7e,0x60,0x00,0x12,0x6a,0x7e,0x60,0x01,
0x9b,0x7b,0x7e,0x82,0x00,0x00,0x00,0x55,0x0a,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xb7,0x0f,0x00,0x00,0x00,0x00,0x00,0x88,0xf4,0x0b,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x61,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x04,0x00,0xd8,0x8f,0xff,0x3d,0xf8,0x48,0xfa,0x3f,0x06,0x00,0x00,0x19,0x00,0x00,0x00,0x0a,0xe0,0x0f,0x22,0x00,0x00,0x00,0xf8,0x84,0x2f,0x40,0x00,0x00,
0x0a,0x80,0xff,0xef,0x01,0x00,0x0c,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x00,0xe0,0x01,0xff,0x3f,0x82,0x00,0x01,
0x7d,0x5e,0x00,0x00,0x80,0xff,0x00,0x00,0x00,0x00,0xbe,0x19,0x00,0x00,0x00,0x00,0x00,0x00,0xe0,0xff,0x0f,0xfe,0x7f,0x00,0x18,0x00,0x00,0x00,0xe0,0x01,0x00,0x00,
0xc0,0xfd,0xbf,0x95,0x03,0x00,0x00,0x00,0x00,0x00,0x80,0x81,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x06,0x00,0xc0,0x3f,0x12,0x8c,0x04,0x00,0x60,0xc8,
0x2f,0xf8,0xe7,0xf9,0xff,0x7f,0xff,0xff,0x07,0xff,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xff,0x03,0xfc,0x6f,0x00,0xc7,0x9f,0x00,0x03,0x80,0xf9,0xfa,0x0f,0x80,
0x7b,0x80,0x37,0x24,0x0c,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x23,0x8c,
0x0d,0x00,0x00,0x01,0x98,0x04,0x00,0x30,0x00,0x00,0x00,0x00,0x06,0x00,0xf2,0x07,0x00,0x44,0x00,0xf8,0x1f,0x20,0x73,0x7e
};

static int mdm_need_parse_data = 0;
ssize_t mdm_send_cmd(int fd, const unsigned char *buf, size_t size) {
    size_t wc = 0;

    while (wc < size) {
        size_t flag = wc;
        const unsigned char *cur = buf + wc;
        unsigned short len = cur[2] + (((unsigned short)cur[3]) << 8) + 5;

        if (cur[0] == 0x7e && cur[1] == 0x01 && (wc + len) <= size && cur[len - 1] == 0x7e) {
            flag += (len - 1);
        }
        else {
            if (flag == 0 && buf[flag] == 0x7E)
                flag++;

            while (buf[flag] != 0x7E && flag < size)
                flag++;
        }

        if (buf[flag] == 0x7E || flag == size) {
#if 0
            size_t nbytes = 0;
            for (nbytes = 0; nbytes < (flag - wc + 1); nbytes++) {
                printf("0x%02X,", buf[wc + nbytes]);
            }
            printf("\n");
#endif

            g_mdm_req = &buf[wc];
            qlog_poll_write(fd, buf + wc, flag - wc + 1, 1000);

            if (mdm_need_parse_data) {
                int rx_wait = 1000;

                while (rx_wait-- > 0) {
                    if (g_mdm_req == NULL)
                        break;
                    usleep(2*1000);
                }
                if (g_mdm_req != NULL) {
                    g_mdm_req = NULL;
                    qlog_dbg("timeout g_mdm_req=%02x\n", buf[wc]);
                }
            }
        }
        else {
            printf("%s unknow mdm cmd\n", __func__);
        }

        if (tty2tcp_sockfd > 0 && (flag + 1) < size) {
            size_t i = 0;
            qlog_dbg("size=%zd, cur=%zd\n", size, flag - wc + 1);
            for (i = 0; i < 16; i++)
                printf("%02x", buf[i+wc]);
            printf("\n");
        }

        wc = flag + 1;
    }

    return size;
}

static int mdm_init_filter(int fd, const char *cfg) {
    unsigned char *rbuf;
    const size_t rbuf_size = (16*1024);
    size_t cfg_size = 0;

#if 1
    if (cfg && !strcmp(cfg, "dump")) {
        const unsigned char mdm_enter_dump1[] = {
            0x4b, 0x12, 0x18, 0x02, 0x01, 0x00, 0xd2, 0x7e
        };
        const unsigned char mdm_enter_dump2[] = {
            0x7e, 0x01, 0x04, 0x00, 0x4b, 0x25, 0x03, 0x00, 0x7e
        };
        qlog_dbg("send mdm dump command\n");
        mdm_send_cmd(fd, mdm_enter_dump1, sizeof(mdm_enter_dump1));
        usleep(100*1000);
        mdm_send_cmd(fd, mdm_enter_dump2, sizeof(mdm_enter_dump2));
        return 0;
    }
#endif

    mdm_need_parse_data = 1;
    mdm_create_qshrink4_file(fd);

    rbuf = (unsigned char *)malloc(rbuf_size);
    if (rbuf == NULL) {
        qlog_dbg("Fail to malloc rbuf_size=%zd, errno: %d (%s)\n", rbuf_size, errno, strerror(errno));
        mdm_need_parse_data = 0;
        return -1;
    }

    if (cfg) {
        int cfgfd = open(cfg, O_RDONLY);
        if (cfgfd < 0) {
            qlog_dbg("Fail to open %s, errno : %d (%s)\n", cfg, errno, strerror(errno));
        }
        else {
            cfg_size = read(cfgfd, rbuf, rbuf_size);
            close(cfgfd);
        }
    }

    if (cfg_size <= 0) {
        cfg_size = sizeof(qlog_mdm_default_cfg);
        memcpy(rbuf, qlog_mdm_default_cfg, cfg_size);
    }

    mdm_send_cmd(fd, rbuf, cfg_size);

    free(rbuf);

    mdm_need_parse_data = 0;
    return 0;
}

static int mdm_clean_filter(int fd) {
    mdm_send_empty_mask();
    return 0;
}

static int mdm_miss_qmdlv2_logfd = -1;
static int mdm_logfile_init(int logfd, unsigned logfile_seq) {
    if (!use_qmdl2_v2)
        return 0;

    if (qlog_le32(qshrink4_data.header_length) == 0) {
        mdm_miss_qmdlv2_logfd = logfd;
        return 0;
    }

    mdm_miss_qmdlv2_logfd = -1;
    write(logfd, &qshrink4_data, qlog_le32(qshrink4_data.header_length));

    return 0;
}

static size_t mdm_logfile_save(int logfd, const void *buf, size_t size) {
    if (mdm_need_parse_data)
        mdm_parse_data_for_command_rsp(buf, size);

    if (use_qmdl2_v2) {
        if (qlog_le32(qshrink4_data.header_length) == 0)
            return size;
        else if (mdm_miss_qmdlv2_logfd == logfd) {
            mdm_miss_qmdlv2_logfd = -1;
            write(logfd, &qshrink4_data, qlog_le32(qshrink4_data.header_length));
        }
    }

    return write(logfd, buf, size);
}

qlog_ops_t mdm_qlog_ops = {
    .init_filter = mdm_init_filter,
    .clean_filter = mdm_clean_filter,
    .logfile_init = mdm_logfile_init,
    .logfile_save = mdm_logfile_save,
};
